﻿using System.ServiceModel;
using BMS.DataContracts;
using BMS.FaultContracts;
using InfoWorld.HL7.ITS;
using System.Collections.Generic;
using System;
using BMS.Utils;

namespace BMS.ServiceContracts
{
    /// <summary>
    /// BMS Persistence operations (CRUD on BMS acts).
    /// </summary>
    [ServiceContract()]
    public interface IBedManagerOperations
    {
        #region Admission

        /// <summary>
        /// Based on an admission order id, it returns all the data related to that admission order.
        /// </summary>
        /// <param name="admissionOrderId">Admission order id (II type) to select from the BMS database.</param>
        /// <returns>The admission order data.</returns>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(EntityNotFoundException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        AdmissionOrder GetAdmissionOrder(II admissionOrderId);

        /// <summary>
        /// Based on an admission event id, it returns all the data related to that admission event.
        /// </summary>
        /// <param name="admissionEventId">Admission event id (II type) to select from the BMS database.</param>
        /// <returns>The admission event data.</returns>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(EntityNotFoundException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        AdmissionEvent GetAdmissionEvent(II admissionEventId);

        /// <summary>
        /// Returns all the data related to an admission event by patient, vista and event ien entered after admission.
        /// </summary>                
        /// <param name="patientId">The patient id.</param>
        /// <param name="vistaId">The vista id.</param>
        /// <param name="eventIen">The event ien.</param>
        /// <returns>The admission event data.</returns>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(EntityNotFoundException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        AdmissionEvent GetAdmissionEventForPatient(II patientId, II vistaId, string eventIen, DateTime? eventDateTime);

        /// <summary>
        /// Creates an admission order within BMS.
        /// </summary>
        /// <param name="admissionOrder">Admission order data to pass when inserting in the database.</param>
        /// <returns>Id of the newly created admission order.</returns>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        II CreateAdmissionOrder(AdmissionOrder admissionOrder);

        /// <summary>
        /// Creates an admission event within BMS.
        /// </summary>
        /// <param name="admissionEvent">Admission event data to pass when inserting in the database.</param>
        /// <returns>Id of the newly created admission event.</returns>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        II CreateAdmissionEvent(AdmissionEvent admissionEvent);        

        /// <summary>
        /// Updates data for an existing admission order in the BMS database.
        /// </summary>
        /// <param name="admissionOrder">Admission order object sent to update. Based on the admission order id, the database will be updated to reflect the data contained within this object.</param>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(EntityNotFoundException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        void UpdateAdmissionOrder(AdmissionOrder admissionOrder);

        /// <summary>
        /// Updates data for an existing admission event in the BMS database.
        /// </summary>
        /// <param name="admissionEvent">Admission event object sent to update. Based on the admission event id, the database will be updated to reflect the data contained within this object.</param>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(EntityNotFoundException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        void UpdateAdmissionEvent(AdmissionEvent admissionEvent);

        #endregion

        #region Movement

        /// <summary>
        /// Creates a new movement order within BMS. 
        /// </summary>
        /// <param name="movementOrder">Movement order data to pass when inserting in the BMS database.</param>
        /// <returns>Id of the newly created movement order.</returns>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        II CreateMovementOrder(MovementOrder movementOrder);

        /// <summary>
        /// Based on a movement order id, it returns all the data related to that movement order.
        /// </summary>
        /// <param name="movementOrderId">Movement order id (II type) used to select the movement order data from the BMS database.</param>
        /// <returns>The movement order data.</returns>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(EntityNotFoundException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        MovementOrder GetMovementOrder(II movementOrderId);

        /// <summary>
        /// Updates data for an existing movement order in the BMS database.
        /// </summary>
        /// <param name="movementOrder">Movement order object sent to update. Based on the movement order id, the database will be updated to reflect the data contained within this movement order.</param>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(EntityNotFoundException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        void UpdateMovementOrder(MovementOrder movementOrder);

        /// <summary>
        /// Creates a new movement event within BMS. 
        /// </summary>
        /// <param name="movementEvent">Movement event data to pass when inserting in the BMS database.</param>
        /// <returns>Id of the newly created movement event.</returns>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        II CreateMovementEvent(MovementEvent movementEvent);

        /// <summary>
        /// Based on a movement event id, it returns all the data related to that movement event.
        /// </summary>
        /// <param name="movementEventId">Movement event id (II type) used to select the movement event data from the BMS database.</param>
        /// <returns>The movement event data.</returns>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(EntityNotFoundException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        MovementEvent GetMovementEvent(II movementEventId);

        /// <summary>
        /// Updates data for an existing movement event in the BMS database.
        /// </summary>
        /// <param name="movementEvent">Movement event object sent to update. Based on the movement event id, the database will be updated to reflect the data contained within this movement event.</param>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(EntityNotFoundException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        void UpdateMovementEvent(MovementEvent movementEvent);

        #endregion

        #region Discharge

        /// <summary>
        /// Creates a new discharge appointment within BMS. 
        /// </summary>
        /// <param name="dischargeAppointment">Discharge appointment data to pass when inserting in the BMS database.</param>
        /// <returns>Id of the newly created discharge appointment.</returns>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        II CreateDischargeAppointment(DischargeAppointment dischargeAppointment);

        /// <summary>
        /// Based on a discharge appointment id, it returns all the data related to that discharge appointment.
        /// </summary>
        /// <param name="dischargeAppointmentId">Discharge appointment id (II type) used to select the discharge appointment data from the BMS database.</param>
        /// <returns>The discharge appointment data.</returns>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(EntityNotFoundException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        DischargeAppointment GetDischargeAppointment(II dischargeAppointmentId);

        /// <summary>
        /// Gets the discharge intent.
        /// </summary>
        /// <param name="appointmentDateTime">The appointment date time.</param>        
        /// <param name="vistaId">The vista id.</param>
        /// <returns></returns>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(EntityNotFoundException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        DischargeAppointment GetDischargeIntent(DateTime appointmentDateTime, II vistaId);

        /// <summary>
        /// Updates data for an existing discharge appointment in the BMS database.
        /// </summary>
        /// <param name="dischargeAppointment">Discharge appointment object sent to update. Based on the discharge appointment id, the database will be updated to reflect the data contained within this discharge appointment.</param>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(EntityNotFoundException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        void UpdateDischargeAppointment(DischargeAppointment dischargeAppointment);

        /// <summary>
        /// Creates a new discharge order within BMS. 
        /// </summary>
        /// <param name="dischargeOrder">Discharge order data to pass when inserting in the BMS database.</param>
        /// <returns>Id of the newly created discharge order.</returns>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        II CreateDischargeOrder(DischargeOrder dischargeOrder);

        /// <summary>
        /// Based on a discharge order id, it returns all the data related to that discharge order.
        /// </summary>
        /// <param name="dischargeOrderId">Discharge order id (II type) used to select the discharge order data from the BMS database.</param>
        /// <returns>The discharge order data.</returns>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(EntityNotFoundException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        DischargeOrder GetDischargeOrder(II dischargeOrderId);

        /// <summary>
        /// Updates data for an existing discharge order in the BMS database.
        /// </summary>
        /// <param name="dischargeOrder">Discharge order object sent to update. Based on the discharge order id, the database will be updated to reflect the data contained within this discharge order.</param>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(EntityNotFoundException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        void UpdateDischargeOrder(DischargeOrder dischargeOrder);

        /// <summary>
        /// Creates a new discharge event within BMS. 
        /// </summary>
        /// <param name="dischargeEvent">Discharge event data to pass when inserting in the BMS database.</param>
        /// <returns>Id of the newly created discharge event.</returns>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        II CreateDischargeEvent(DischargeEvent dischargeEvent);

        /// <summary>
        /// Based on a discharge event id, it returns all the data related to that discharge event.
        /// </summary>
        /// <param name="dischargeEventId">Discharge event id (II type) used to select the discharge event data from the BMS database.</param>
        /// <returns>The discharge event data.</returns>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(EntityNotFoundException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        DischargeEvent GetDischargeEvent(II dischargeEventId);

        /// <summary>
        /// Updates data for an existing discharge event in the BMS database.
        /// </summary>
        /// <param name="dischargeEvent">Discharge event object sent to update. Based on the discharge event id, the database will be updated to reflect the data contained within this discharge event.</param>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(EntityNotFoundException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        void UpdateDischargeEvent(DischargeEvent dischargeEvent);

        #endregion

        /// <summary>
        /// Creates a new transfer act within BMS. Creation means a tranfer intent will be inserted in the BMS database.
        /// </summary>
        /// <param name="transfer">Transfer data to pass when inserting in the BMS database. Contains data about a transfer intent.</param>
        /// <returns>Id of the newly created transfer.</returns>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        II CreateTransfer(Transfer transfer);

        /// <summary>
        /// Based on a transfer Id it returns all the data related to that transfer.
        /// </summary>
        /// <param name="transferId">Transfer id (II type) used to select the transfer data from the BMS database.</param>
        /// <returns>The transfer data - includes data to the intent, request and event.</returns>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(EntityNotFoundException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        Transfer GetTransfer(II transferId);

        /// <summary>
        /// Updates data for an existing transfer in the BMS database.
        /// </summary>
        /// <param name="transfer">Transfer object sent to update. Based on the transfer id, the database will be updated to reflect the data contained within this transfer act.</param>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(EntityNotFoundException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        void UpdateTransfer(Transfer transfer);                        

        /// <summary>
        /// Creates a new bed cleaning operation within BMS database.
        /// </summary>
        /// <param name="bedCleaningOperation">Bed cleaning operation data to pass when inserting in the BMS database.</param>
        /// <returns>Id of the newly created bed cleaning operation.</returns>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        II CreateBedCleaningOperation(BedCleaningOperation bedCleaningOperation);

        /// <summary>
        /// Based on a bed cleaning operation Id it returns all the data related to that bed cleaning operation.
        /// </summary>
        /// <param name="bedCleaningOperationId">Bed cleaning operation id (II type) used to select the bed cleaning operation data from the BMS database.</param>
        /// <returns>The bed cleaning operation act data.</returns>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(EntityNotFoundException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        BedCleaningOperation GetBedCleaningOperation(II bedCleaningOperationId, II bedId);

        /// <summary>
        /// Updates data for an existing bed cleaning operation in the BMS database.
        /// </summary>
        /// <param name="bedCleaningOperation">Bed cleaning operation object sent to update. Based on the bed cleaning operation id, the database will be updated to reflect the data contained within this bed cleaning operation act.</param>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(EntityNotFoundException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        void UpdateBedCleaningOperation(BedCleaningOperation bedCleaningOperation);

        /// <summary>
        /// Creates a new bed unavailable event within BMS database.
        /// </summary>
        /// <param name="bedUnavailable">Bed unavailable event data to pass when inserting in the BMS database.</param>
        /// <returns>Id of the newly created bed unavailable event.</returns>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        II CreateBedUnavailable(BedUnavailable bedUnavailable);

        /// <summary>
        /// Based on a bed unavailable event Id it returns all the data related to that bed unavailable event.
        /// </summary>
        /// <param name="bedUnavailableId">Bed unavailable event id (II type) used to select the bed unavailable event data from the BMS database.</param>
        /// <param name="bedId">The bed id.</param>
        /// <returns>The bed unavailable event act data.</returns>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(EntityNotFoundException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        BedUnavailable GetBedUnavailable(II bedUnavailableId, II bedId);

        /// <summary>
        /// Updates data for an existing bed unavailable event in the BMS database.
        /// </summary>
        /// <param name="bedUnavailable">Bed unavailable event object sent to update. Based on the bed unavailable event id, the database will be updated to reflect the data contained within this bed unavailable event act.</param>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(EntityNotFoundException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        void UpdateBedUnavailable(BedUnavailable bedUnavailable);

        /// <summary>
        /// Creates a new diversion entry for BMS.
        /// </summary>
        /// <param name="diversion">Diversion data to insert in the BMS database.</param>
        /// <returns>Diversion id (II type) of the newly created diversion entry.</returns>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        II CreateDiversion(Diversion diversion);

        /// <summary>
        /// Gets data for an existing diversion entry.
        /// </summary>
        /// <param name="diversionId">Based on this id, it selects data from the diversion table within BMS database.</param>
        /// <returns>Diversion data.</returns>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(EntityNotFoundException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        Diversion GetDiversion(II diversionId);

        /// <summary>
        /// Gets data for all diversions.
        /// </summary>
        /// <returns></returns>
        [OperationContract]
        [FaultContract(typeof(EntityNotFoundException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        IList<Diversion> GetAllDiversion(II facilityId);

        /// <summary>
        /// Updates an existing diversion based on the diversion id. 
        /// </summary>
        /// <param name="diversion">Diversion data object sent to the database to update an existing diversion entry.</param>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(EntityNotFoundException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        void UpdateDiversion(Diversion diversion);        

        /// <summary>
        /// Adds a patient waiting within BMS.
        /// </summary>
        /// <param name="waitingListItem">The waiting list item.</param>
        /// <returns>
        /// Id of the newly created waiting list item.
        /// </returns>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        II CreateWaitingListItem(WaitingListItem waitingListItem);

        /// <summary>
        /// Based on a waiting list item Id it returns all the related data.
        /// </summary>
        /// <param name="waitingListItemId">The waiting list item id.</param>
        /// <returns>
        /// The waiting list item data.
        /// </returns>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(EntityNotFoundException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        WaitingListItem GetWaitingListItem(II waitingListItemId, string patientUid);

        /// <summary>
        /// Updates data for an existing waiting list item in the BMS database.
        /// </summary>
        /// <param name="admission">Waiting list item object sent to update. Based on the admission id, the database will be updated to reflect the data contained within this object.</param>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(EntityNotFoundException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        void UpdateWaitingListItem(WaitingListItem waitingListItem);

        /// <summary>
        /// Inserts the bed unvailable history.
        /// </summary>
        /// <param name="bed">The bed.</param>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(EntityNotFoundException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        void InsertBedUnavailableHistory(BedUnavailableHistory bedUnavailableHistory);

        /// <summary>
        /// Creates the icon detail.
        /// </summary>
        /// <param name="iconDetail">The icon detail.</param>
        /// <returns></returns>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        int CreateIconDetail(IconDetail iconDetail);

        /// <summary>
        /// Gets the icon detail.
        /// </summary>
        /// <param name="iconDetailId">The icon detail id.</param>
        /// <param name="facility">The facility.</param>
        /// <returns></returns>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        IconDetail GetIconDetail(int iconDetailId, II facilityId);

        /// <summary>
        /// Updates the icon detail.
        /// </summary>
        /// <param name="iconDetail">The icon detail.</param>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(EntityNotFoundException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        void UpdateIconDetail(IconDetail iconDetail);

        /// <summary>
        /// Deletes the icon detail.
        /// </summary>
        /// <param name="iconDetailId">The icon detail id.</param>
        /// <param name="facility">The facility.</param>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        void DeleteIconDetail(int iconDetailId, II facilityId);

        /// <summary>
        /// Inserts the icon association.
        /// </summary>
        /// <param name="iconAssociation">The icon association.</param>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        void InsertIconAssociation(IconAssociation iconAssociation);

        /// <summary>
        /// Deletes the icon association.
        /// </summary>
        /// <param name="patientUID">The patient UID.</param>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        void DeleteIconAssociation(Guid patientUID, int? iconFlagId);

        /// <summary>
        /// Switches the icon order.
        /// </summary>
        /// <param name="icon1">The icon1.</param>
        /// <param name="icon2">The icon2.</param>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        void SwitchIconOrder(IconInfo icon1, IconInfo icon2);

        /// <summary>
        /// Updates the icon order for a pre-sorted array of icon ids.
        /// </summary>
        /// <param name="iconOrderList">The icon order list.</param>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        void UpdateIconOrder(List<DataContracts.IconOrder> iconOrderList);

        /// <summary>
        /// Updates the icon order bulk.
        /// </summary>
        /// <param name="iconOrderList">The icon order list.</param>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        void UpdateIconOrderBulk(List<IconOrder> iconOrderList);

        /// <summary>
        /// Gets the icon detail.
        /// </summary>
        /// <param name="patientId">The patient id.</param>
        /// <returns></returns>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        PatientLocation GetPatientLocation(II patientId);

        /// <summary>
        /// Creates the bed switch.
        /// </summary>
        /// <param name="bedSwitch">The bed switch.</param>
        /// <returns></returns>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        II CreateBedSwitch(BedSwitch bedSwitch);

        /// <summary>
        /// Deletes the bed staff.
        /// </summary>
        /// <param name="bedId">The bed id.</param>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        void DeleteBedStaff(II bedId);

        /// <summary>
        /// Inserts the bed staff.
        /// </summary>
        /// <param name="bedStaff">The bed staff.</param>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        void CreateBedStaff(BedStaff bedStaff);

        /// <summary>
        /// Gets the bed staff.
        /// </summary>
        /// <param name="bedId">The bed id.</param>
        /// <returns></returns>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        BedStaff GetBedStaff(II bedId);

        /// <summary>
        /// Gets the logged users.
        /// </summary>
        /// <param name="domainId">The domain id.</param>
        /// <param name="facilityIdList">The facility id list.</param>
        /// <returns></returns>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        List<LoggedUser> GetLoggedUsers(string domainId, string facilityIdList);

        /// <summary>
        /// Inserts the logged users.
        /// </summary>
        /// <param name="user">The user.</param>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        void InsertLoggedUsers(LoggedUser user);

        /// <summary>
        /// Clears the logged users.
        /// </summary>
        /// <param name="domainId">The domain id.</param>
        [OperationContract]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        void ClearLoggedUsers(string domainId);

        /// <summary>
        /// Deletes the ems staff.
        /// </summary>
        /// <param name="emsStaffId">The ems staff id.</param>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        void DeleteEmsStaff(II emsStaffId);

        /// <summary>
        /// Creates the ems staff.
        /// </summary>
        /// <param name="emsStaff">The ems staff.</param>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        void CreateEmsStaff(EmsStaff emsStaff);

        /// <summary>
        /// Updates the ems staff.
        /// </summary>
        /// <param name="emsStaff">The ems staff.</param>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        void UpdateEmsStaff(EmsStaff emsStaff);

        /// <summary>
        /// Gets the ems staff.
        /// </summary>
        /// <param name="emsStaffId">The ems staff id.</param>
        /// <returns></returns>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        EmsStaff GetEmsStaff(II emsStaffId);

        /// <summary>
        /// Adds the log entry.
        /// </summary>
        /// <param name="logEntry">The log entry.</param>
        [OperationContract]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        int AddLogEntry(JobLogInfo logEntry);

        /// <summary>
        /// Updates the log entry.
        /// </summary>
        /// <param name="logEntry">The log entry.</param>
        [OperationContract]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        void UpdateLogEntry(JobLogInfo logEntry);

        /// <summary>
        /// Gets the data.
        /// </summary>
        /// <param name="site">The site.</param>
        /// <returns></returns>
        [OperationContract]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        IList<VistaOperation> GetData(string vistaId);

        /// <summary>
        /// Updates the data.
        /// </summary>
        /// <param name="site">The site.</param>
        /// <param name="type">The type.</param>
        /// <param name="lastRunning">The last running.</param>
        [OperationContract]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        void UpdateData(VistaOperation vistaOperation);

        /// <summary>
        /// Deletes the data.
        /// </summary>
        /// <param name="site">The site.</param>
        [OperationContract]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        void DeleteData(string vistaId);

        /// <summary>
        /// Saves the level of care.
        /// </summary>
        /// <param name="data">The data.</param>
        [OperationContract]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        void SaveLevelOfCare(LevelOfCareData data);

        /// <summary>
        /// Saves the vista integration log.
        /// </summary>
        /// <param name="vistaIntegrationLog">The vista integration log.</param>
        [OperationContract]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        void SaveVistaIntegrationLog(VistaIntegrationLog vistaIntegrationLog, string vista);

        /// <summary>
        /// Creates the specialty transfer.
        /// </summary>
        /// <param name="specialtyTransfer">The specialty transfer.</param>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        void CreateSpecialtyTransfer(SpecialtyTransfer specialtyTransfer);

        /// <summary>
        /// Gets the specialty transfer.
        /// </summary>
        /// <param name="specialtyTransferId">The specialty transfer id.</param>
        /// <returns></returns>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(EntityNotFoundException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        SpecialtyTransfer GetSpecialtyTransfer(II specialtyTransferId);

        /// <summary>
        /// Updates the specialty transfer.
        /// </summary>
        /// <param name="specialtyTransfer">The specialty transfer.</param>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(EntityNotFoundException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        void UpdateSpecialtyTransfer(SpecialtyTransfer specialtyTransfer);

        /// <summary>
        /// Creates the hospital location appointment.
        /// </summary>
        /// <param name="hospitalLocationAppointment">The hospital location appointment.</param>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        void CreateHospitalLocationAppointment(HospitalLocationAppointment hospitalLocationAppointment);

        /// <summary>
        /// Gets the hospital locations appointment.
        /// </summary>
        /// <param name="facilityUid">The facility uid.</param>
        /// <param name="vistaSiteUid">The vista site uid.</param>
        /// <returns></returns>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(EntityNotFoundException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        List<HospitalLocationAppointment> GetHospitalLocationsAppointment(Guid? facilityUid, Guid? vistaSiteUid);

        /// <summary>
        /// Gets the hospital location appointment.
        /// </summary>
        /// <param name="hospitalLocationAppointmentId">The hospital location appointment id.</param>
        /// <returns></returns>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(EntityNotFoundException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        HospitalLocationAppointment GetHospitalLocationAppointment(II hospitalLocationAppointmentId);

        /// <summary>
        /// Deletes the hospital location appointment.
        /// </summary>
        /// <param name="hospitalLocationAppointmentId">The hospital location appointment id.</param>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(EntityNotFoundException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        void DeleteHospitalLocationAppointment(II hospitalLocationAppointmentId);

        /// <summary>
        /// Creates a scheduled admission within BMS.
        /// </summary>
        /// <param name="admission">Admission data to pass when inserting in the database.</param>
        /// <returns>Id of the newly created scheduled admission.</returns>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(EntityNotFoundException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        II CreateScheduledAdmission(ScheduledAdmission scheduledAdmission);

        /// <summary>
        /// Based on a scheduled admission Id it returns all the data related to that scheduled admission.
        /// </summary>
        /// <param name="admissionId">Scheduled admission Id (II type) to select from the BMS database.</param>
        /// <returns>The scheduled admission data.</returns>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(EntityNotFoundException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        ScheduledAdmission GetScheduledAdmission(II scheduledAdmissionId);

        /// <summary>
        /// Gets the scheduled admission.
        /// </summary>
        /// <param name="site">The site.</param>
        /// <param name="ien">The ien.</param>
        /// <returns></returns>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(EntityNotFoundException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        DataContracts.ScheduledAdmission GetScheduledAdmissionByIen(II vistaId, string ien);

        /// <summary>
        /// Updates data for an existing scheduled admission in the BMS database.
        /// </summary>
        /// <param name="admission">Scheduled admission object sent to update. Based on the scheduled admission id, the database will be updated to reflect the data contained within this object.</param>
        [OperationContract]
        [FaultContract(typeof(NullParameterException))]
        [FaultContract(typeof(EntityNotFoundException))]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        void UpdateScheduledAdmission(ScheduledAdmission scheduledAdmission);

        /// <summary>
        /// Manages the deleted movements.
        /// </summary>
        /// <param name="iens">The iens.</param>
        /// <param name="vistaId">The vista id.</param>
        [OperationContract]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        void ManageDeletedMovements(string iens, II vistaId, string vistaCode);

        /// <summary>
        /// Updates the movements without bed(bulk).
        /// </summary>
        /// <param name="iens">The "iens" parameter contains multiple movements data with the format: ien, ward_uid, bed_uid, ien, ward_uid, bed_uid, .....</param>
        /// <param name="vistaId">The vista id.</param>
        [OperationContract]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        void UpdateMovementsWithoutBed(string iens, II vistaId);

		/// <summary>
		/// Updates audit jobs failed with status running to failed.
		/// </summary>
		[OperationContract]
		[FaultContract(typeof(CustomSQLException))]
		[FaultContract(typeof(OtherException))]
		void UpdateAuditJobsFailedWithStatusRunningToFailed();

        /// <summary>
        /// Gets the vista patient ien.
        /// </summary>
        /// <param name="vistaCode">The vista code.</param>
        /// <returns></returns>
        [OperationContract]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        string GetVistaPatientIen(string vistaCode);
	}
}
